<?php

namespace AppBundle\Entity;

use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Doctrine\ORM\Mapping\OrderBy;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Utilisateur
 *
 * @ORM\Table(name="utilisateur")
 * @ORM\Entity(repositoryClass="AppBundle\Entity\Repositories\UtilisateurRepository")
 */
class Utilisateur implements UserInterface
{
    /**
     * @var string
     *
     * @Assert\NotBlank()
     * @ORM\Column(name="nom", type="string", length=45, nullable=false)
     */
    private $nom;

    /**
     * @var string
     *
     * @Assert\NotBlank()
     * @ORM\Column(name="prenom", type="string", length=45, nullable=false)
     */
    private $prenom;

    /**
     * @var string
     *
     * @Assert\NotBlank()
     * @Assert\Email()
     * @ORM\Column(name="email", type="string", length=45, nullable=false)
     */
    private $email;

    /**
     * @var string
     *
     * @Assert\NotBlank()
     * @Assert\Length(min = 10, max = 10)
     * @ORM\Column(name="telephone", type="string", length=45, nullable=false)
     */
    private $telephone;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="date_creation", type="datetime", nullable=false)
     */
    private $dateCreation;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="date_modification", type="datetime", nullable=false)
     */
    private $dateModification;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="date_fin_ban", type="datetime", nullable=false)
     */
    private $dateFinBan;

    /**
     * @var boolean
     *
     * @ORM\Column(name="is_actif", type="boolean", nullable=false)
     */
    private $isActif;

    /**
     * @var boolean
     *
     * @ORM\Column(name="is_agent", type="boolean", nullable=false)
     */
    private $isAgent;

    /**
     * @var boolean
     *
     * @ORM\Column(name="is_admin", type="boolean", nullable=false)
     */
    private $isAdmin;

    /**
     * @var boolean
     *
     * @ORM\Column(name="is_superadmin", type="boolean", nullable=false)
     */
    private $isSuperadmin;

    /**
     * @var boolean
     *
     * @ORM\Column(name="is_ban", type="boolean", nullable=false)
     */
    private $isBan;

    /**
     * @var string
     *
     * @ORM\Column(name="mdp", type="string", length=45, nullable=false)
     */
    private $mdp;

    private $plainPassword;

    /**
     * @var boolean
     *
     * @ORM\Column(name="has_validated", type="boolean", nullable=false)
     */
    private $hasValidated;

    /**
     * @var string
     *
     * @ORM\Column(name="hash", type="string", nullable=true)
     */
    private $hash;

    /**
     * @var integer
     *
     * @ORM\Column(name="id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY")
     */
    private $id;

    /**
     * @var \Doctrine\Common\Collections\Collection
     *
     * @ORM\ManyToMany(targetEntity="AppBundle\Entity\Interlocuteur", inversedBy="utilisateur")
     * @ORM\JoinTable(name="utilisateur_interlocuteur",
     *   joinColumns={
     *     @ORM\JoinColumn(name="utilisateur_id", referencedColumnName="id")
     *   },
     *   inverseJoinColumns={
     *     @ORM\JoinColumn(name="interlocuteur_siret", referencedColumnName="siret")
     *   }
     * )
     * @OrderBy({"identifiant" = "ASC"})
     */
    private $interlocuteurSiret;

    /**
     * @var \Doctrine\Common\Collections\Collection
     *
     * @ORM\ManyToMany(targetEntity="AppBundle\Entity\Thematique", inversedBy="utilisateur")
     * @ORM\JoinTable(name="utilisateur_thematique",
     *   joinColumns={
     *     @ORM\JoinColumn(name="utilisateur_id", referencedColumnName="id")
     *   },
     *   inverseJoinColumns={
     *     @ORM\JoinColumn(name="thematique_code", referencedColumnName="code")
     *   }
     * )
     * @OrderBy({"intitule" = "ASC"})
     */
    private $thematiqueCode;

    /**
     * Constructor
     */
    public function __construct($data = null)
    {
        $this->interlocuteurSiret = new \Doctrine\Common\Collections\ArrayCollection();
        $this->thematiqueCode = new \Doctrine\Common\Collections\ArrayCollection();

        if($data){
            $this->email = $data['email'];
            $this->nom = $data['nom'];
            $this->prenom = $data['prenom'];
            $this->telephone = $data['telephone'];
            $this->dateCreation = new DateTime();
            $this->dateModification = new DateTime();
            $this->isActif = true;
            $this->isAgent = false;
            $this->isAdmin = false;
            $this->isSuperadmin = false;
            $this->isBan = false;
            $this->hasValidated = false;
            $this->plainPassword = $data['password'];
            $this->hash = sha1(uniqid());
        }
    }

    public function getAgences()
    {
        $agences = new ArrayCollection();

        /** @var Interlocuteur $interloc */
        foreach ($this->getInterlocuteurSiret() as $interloc) {
            if (!$agences->contains($interloc->getCodeAgence()))
                $agences->add($interloc->getCodeAgence());
        }

        return $agences;
    }

    public function addThematique(Thematique $thematique)
    {
        $this->thematiqueCode->add($thematique);
    }

    public function getThematiqueCodes()
    {
        $keys = [];
        foreach ($this->thematiqueCode as $them) {
            $keys[] = $them->getCode();
        }

        return $keys;
    }

    public function getInterlocuteurSiretList()
    {
        $keys = [];
        foreach ($this->interlocuteurSiret as $inter) {
            $keys[] = $inter->getSiret();
        }

        return $keys;
    }

    public function emptyThematique()
    {
        foreach ($this->thematiqueCode as $them) {
            $this->thematiqueCode->removeElement($them);
        }
    }

    public function addInterlocuteur(Interlocuteur $interlocuteur)
    {
        $this->interlocuteurSiret->add($interlocuteur);
    }

    public function emptyInterlocuteur()
    {
        foreach ($this->interlocuteurSiret as $inter) {
            $this->interlocuteurSiret->removeElement($inter);
        }
    }

    public function getHash()
    {
        return $this->hash;
    }

    public function setHash($hash)
    {
        $this->hash = $hash;
    }

    public function getUsername()
    {
        return $this->email;
    }

    public function getRoles()
    {
        $roles = [
            'ROLE_USER'
        ];

        if($this->isAgent) {
            $roles[] = 'ROLE_AGENT';
        }
        if($this->isAdmin) {
            $roles[] = 'ROLE_ADMIN';
        }
        if($this->isAdmin or $this->isAgent) {
            $roles[] = 'ROLE_ALLOWED_TO_SWITCH';
        }

        return $roles;
    }

    public function getPassword()
    {
        return $this->mdp;
    }

    public function getSalt()
    {
        // TODO: Implement getSalt() method.
    }

    public function eraseCredentials()
    {
        $this->plainPassword = null;
    }

    /**
     * @return string
     */
    public function getNom()
    {
        return $this->nom;
    }

    /**
     * @param string $nom
     */
    public function setNom($nom)
    {
        $this->nom = $nom;
    }

    /**
     * @return string
     */
    public function getPrenom()
    {
        return $this->prenom;
    }

    /**
     * @param string $prenom
     */
    public function setPrenom($prenom)
    {
        $this->prenom = $prenom;
    }

    /**
     * @return string
     */
    public function getEmail()
    {
        return $this->email;
    }

    /**
     * @param string $email
     */
    public function setEmail($email)
    {
        $this->email = $email;
    }

    /**
     * @return string
     */
    public function getTelephone()
    {
        return $this->telephone;
    }

    /**
     * @param string $telephone
     */
    public function setTelephone($telephone)
    {
        $this->telephone = $telephone;
    }

    /**
     * @return \DateTime
     */
    public function getDateCreation()
    {
        return $this->dateCreation;
    }

    /**
     * @param \DateTime $dateCreation
     */
    public function setDateCreation($dateCreation)
    {
        $this->dateCreation = $dateCreation;
    }

    /**
     * @return \DateTime
     */
    public function getDateModification()
    {
        return $this->dateModification;
    }

    /**
     * @param \DateTime $dateModification
     */
    public function setDateModification($dateModification)
    {
        $this->dateModification = $dateModification;
    }

    /**
     * @return string
     */
    public function getDateFinBan()
    {
        return $this->dateFinBan ? $this->dateFinBan->format('d/m/Y') : null;
    }

    public function getDateFinBanDT()
    {
        return $this->dateFinBan;
    }

    /**
     * @param \DateTime $dateFinBan
     */
    public function setDateFinBan($dateFinBan)
    {
        if($dateFinBan instanceof DateTime) {
            $this->dateFinBan = $dateFinBan;
        } else {
            $date = DateTime::createFromFormat('d/m/Y', $dateFinBan);
            $this->dateFinBan = $date ? $date : null;
        }
    }

    /**
     * @return bool
     */
    public function isIsActif()
    {
        return $this->isActif;
    }

    /**
     * @param bool $isActif
     */
    public function setIsActif($isActif)
    {
        $this->isActif = $isActif;
    }

    /**
     * @return bool
     */
    public function isIsAgent()
    {
        return $this->isAgent;
    }

    /**
     * @param bool $isAgent
     */
    public function setIsAgent($isAgent)
    {
        $this->isAgent = $isAgent;
    }

    /**
     * @return bool
     */
    public function isIsAdmin()
    {
        return $this->isAdmin;
    }

    /**
     * @param bool $isAdmin
     */
    public function setIsAdmin($isAdmin)
    {
        $this->isAdmin = $isAdmin;
    }

    /**
     * @return bool
     */
    public function isIsSuperadmin()
    {
        return $this->isSuperadmin;
    }

    /**
     * @param bool $isSuperadmin
     */
    public function setIsSuperadmin($isSuperadmin)
    {
        $this->isSuperadmin = $isSuperadmin;
    }

    /**
     * @return bool
     */
    public function isIsBan()
    {
        return $this->isBan;
    }

    /**
     * @param bool $isBan
     */
    public function setIsBan($isBan)
    {
        $this->isBan = $isBan;
    }

    /**
     * @return bool
     */
    public function isHasValidated()
    {
        return $this->hasValidated;
    }

    /**
     * @param bool $hasValidated
     */
    public function setHasValidated($hasValidated)
    {
        $this->hasValidated = $hasValidated;
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getInterlocuteurSiret()
    {
        return $this->interlocuteurSiret;
    }

    /**
     * @param \Doctrine\Common\Collections\Collection $interlocuteurSiret
     */
    public function setInterlocuteurSiret($interlocuteurSiret)
    {
        $this->interlocuteurSiret = $interlocuteurSiret;
    }

    /**
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getThematique()
    {
        return $this->thematiqueCode;
    }

    /**
     * @param \Doctrine\Common\Collections\Collection $thematiqueCode
     */
    public function setThematiqueCode($thematiqueCode)
    {
        $this->thematiqueCode = $thematiqueCode;
    }

    /**
     * @return mixed
     */
    public function getPlainPassword()
    {
        return $this->plainPassword;
    }

    /**
     * @param mixed $plainPassword
     */
    public function setPlainPassword($plainPassword)
    {
        $this->plainPassword = $plainPassword;
        $this->mdp = null;
    }

    /**
     * @param string $mdp
     */
    public function setMdp($mdp)
    {
        $this->mdp = $mdp;
    }

    /**
     * @return string
     */
    public function getMdp()
    {
        return $this->mdp;
    }

    public function setPassword($string)
    {

    }


}

